package gov.va.med.domain.typecode;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.util.Assert;

/*
 * This class implements an abstraction of the type-safe enumeration pattern, exposing methods that
 * return sorted varieties of the instances of a specific concrete type. All subclasses should
 * declare private constructors. 
 * @author PII Apr 12, 2005 9:19:58 AM
 */
public abstract class TypeCodeAbstract implements TypeCode {
	private static Logger logger = LogManager.getLogger(TypeCodeAbstract.class);
	private Object code;
	private String description;
	private String label;
	private static Map typeCache = new HashMap();

	protected TypeCodeAbstract(Object code, String label, String description) {
		setCode(code);
		setLabel(label);
		setDescription(description);
		putCode(this);
	}

	protected TypeCodeAbstract(Object code, String label) {
		this(code, label, label);
	}

	/**
	 * Return a type code instance for a given class type that maps back to a given code value.
	 * 
	 * @param value
	 * @return TypeCodeAbstract
	 */
	public static TypeCode codeForValue(Class target, Object value) {
		return fetchCodeCache(target).findCode(value);
	}

	/**
	 * Return an array of type codes, sorted by their codes.
	 * @param type
	 * @return TypeCode[]
	 */
	public static TypeCode[] sortByCode(Class type) {
		return fetchCodeCache(type).sortByCode();
	}

	/**
	 * Return an array of type codes, sorted by their labels.
	 * @param type
	 * @return TypeCode[]
	 */
	public static TypeCode[] sortByLabel(Class type) {
		return fetchCodeCache(type).sortByLabel();
	}

	/**
	 * Return a boolean indicator of whether an instance of the given type code class maps to the
	 * specified code value.
	 * @param type
	 * @param codeValue
	 * @return boolean
	 */
	public static boolean isDefined(Class type, Object codeValue) {
		return fetchCodeCache(type).contains(codeValue);
	}

	/*
	 * Compare to another TypeCodeAbstract instance.
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Object other) {
		return ((Comparable)getCode()).compareTo(((TypeCodeAbstract)other).getCode());
	}

	/**
	 * Return the code of this type code instance.
	 * 
	 * @return Object
	 */
	public Object getCode() {
		return this.code;
	}

	/**
	 * Return the description of this type code instance.
	 * 
	 * @return Returns the description.
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * Return the label of this typecode instance.
	 * 
	 * @return Returns the label.
	 */
	public String getLabel() {
		return label;
	}

	/*
	 * Override superclass equals to declare as final. This prevents subclasses from breaking the
	 * equals contract and allows for safe identity comparison (with "==").
	 * 
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public final boolean equals(Object obj) {
		return super.equals(obj);
	}

	/*
	 * Override superclass hashCode to declare as final. This prevents subclasses from breaking the
	 * hashCode contract.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	public final int hashCode() {
		return super.hashCode();
	}

	/**
	 * Cache a typeCode instance in its appropriate cache, keyed by its code value.
	 * 
	 * @param typeCode
	 */
	private static void putCode(TypeCode typeCode) {
		CodeCache codeCache = fetchCodeCache(typeCode.getClass());
		codeCache.putCode(typeCode);
	}

	private static CodeCache fetchCodeCache(Class target) {
		Assert.notNull(target, "Target class cannot be null.");
		Assert.isTrue(TypeCode.class.isAssignableFrom(target), "Target class [" + target.getName()
				+ "] does not implement " + TypeCode.class.getName());
		
		CodeCache codeCache = (CodeCache)typeCache.get(target);

		if(codeCache == null) {
			codeCache = new CodeCache(target);
			typeCache.put(target, codeCache);
		}

		return codeCache;
	}

	/**
	 * Set the code of this instance. Private, as it should be set only on construction.
	 * 
	 * @param code The code to set.
	 */
	private void setCode(Object code) {
		Assert.notNull(code, "Code value for a type code cannot be null.");
		this.code = code;
	}

	/**
	 * Set the description of this instance. Private, as it should be set only on construction.
	 * 
	 * @param description
	 */
	private void setDescription(String description) {
		Assert.notNull(description, "Description value for a type code cannot be null.");
		this.description = description;
	}

	/**
	 * Set the labe of this instance. Private, as it should be set only on construction.
	 * 
	 * @param label
	 */
	private void setLabel(String label) {
		Assert.notNull(label, "Label value for a type code cannot be null.");
		this.label = label;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return getClass().getName() + ": code: " + getCode() + "; label: " + getLabel()
				+ "; description: " + getDescription();
	}

	/**
	 * Inner class for cacheing of enumerated type instances.
	 */
	private static class CodeCache {
		private Map codes;
		private Class parent;

		private CodeCache(Class parent) {
			this.parent = parent;
			codes = new HashMap();
		}

		private TypeCode getCode(Object codeValue) {
			return (TypeCode)codes.get(codeValue);
		}

		private TypeCode findCode(Object codeValue) {
			TypeCode typeCode = (TypeCode)codes.get(codeValue);

			if(typeCode == null) {
				if(logger.isDebugEnabled())
					logger.debug(this);

				throw new NoSuchElementException("No instance of type code ["
						+ parent.getName() + "] exists with code value " + codeValue);
			}

			return typeCode;
		}

		private boolean contains(Object codeValue) {
			return codes.containsKey(codeValue);
		}

		private synchronized void putCode(TypeCode typeCode) {
			if(codes.containsKey(typeCode.getCode()))
				throw new IllegalArgumentException("TypeCode instance of type " + typeCode.getClass().getName() + " with code value " + typeCode.getCode());
			codes.put(typeCode.getCode(), typeCode);
		}

		private TypeCode[] sortByCode() {
			List codeList = new ArrayList(codes.values());
			Collections.sort(codeList);
			return (TypeCode[])codeList.toArray(new TypeCode[codeList.size()]);
		}

		private TypeCode[] sortByLabel() {
			List codeList = new ArrayList(codes.values());
			Collections.sort(codeList, new Comparator() {
				public int compare(Object first, Object second) {
					String firstLabel = ((TypeCode)first).getLabel();
					String secondLabel = ((TypeCode)second).getLabel();
					return firstLabel.compareTo(secondLabel);
				}
			});
			return (TypeCode[])codeList.toArray(new TypeCode[codeList.size()]);
		}
		
		/* (non-Javadoc)
		 * @see java.lang.Object#toString()
		 */
		public String toString() {
			String sep = System.getProperty("line.separator");
			StringBuffer buf = new StringBuffer(getClass().getName()).append(sep);
			buf.append("TypeCode cache for instances of type: " + parent.getName());
			
			for (Iterator i = codes.values().iterator(); i.hasNext();) {
				TypeCode element = (TypeCode)i.next();
				buf.append(sep).append("code: ").append(element.getCode());
				buf.append(sep).append("label: ").append(element.getLabel());
				buf.append(sep).append("desc: ").append(element.getDescription());
				buf.append(sep);
			}
			
			return buf.toString();
		}
	}
}